﻿namespace Hims.Api.Controllers
{
    using System;
    using System.Collections.Generic;
    using System.Threading.Tasks;
    using Domain.Services;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Npgsql;
    using Shared.DataFilters;
    using Shared.EntityModels;
    using Shared.UserModels.Filters;
    using Utilities;
    using Hims.Shared.Library.Enums;
    using Hims.Api.Models;

    /// <inheritdoc />
    [AllowAnonymous]
    [Route("api/lab-order")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class LabOrderController : BaseController
    {
        /// <summary> The lab order service.</summary>
        private readonly ILabOrderService labOrderService;

        /// <summary>
        /// The audit log services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;

        /// <inheritdoc />
        public LabOrderController(ILabOrderService labOrderService, IAuditLogService auditLogServices) { this.labOrderService = labOrderService; this.auditLogServices = auditLogServices; }

        /// <summary>
        /// The fetch LabOrder.
        /// </summary>
        /// <param name="model">
        /// The LabOrder filter model.
        /// </param>
        /// <returns>
        /// The list of LabOrders.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - List of LabOrders.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("fetch")]
        [ProducesResponseType(typeof(List<LabOrderModel>), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAsync([FromBody]LabOrderFilterModel model)
        {
            model = (LabOrderFilterModel)EmptyFilter.Handler(model);
            var labOrders = await this.labOrderService.FetchAsync(model);
            return this.Success(labOrders);
        }

        /// <summary>
        /// The add generalAdvice.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - generalAdvice added successfully.
        /// - 409 - generalAdvice already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("add")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> AddAsync([FromBody] LabOrderModel model, [FromHeader] LocationHeader header)
        {
            model = (LabOrderModel)EmptyFilter.Handler(model);
            var response = await this.labOrderService.AddAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Given Lab Order name has already been exists with us.");
                case 0:
                    return this.ServerError();
            }
            var auditLogModel = new AuditLogModel
            {
                AccountId = model.CreatedBy,
                LogTypeId = (int)LogTypes.LabOrders,
                LogFrom = (int)AccountType.Administrator,
                LogDate = DateTime.UtcNow,
                LogDescription = $"{model.Name} Lab Order has been added successfully.",
                LocationId= Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);
            return this.Success(" Lab Order has been added successfully.");
        }

        /// <summary>
        /// The update generalAdvice.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - generalAdvice updated successfully.
        /// - 409 - generalAdvice already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPut]
        [Route("update")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> UpdateAsync([FromBody] LabOrderModel model, [FromHeader] LocationHeader header)
        {
            model = (LabOrderModel)EmptyFilter.Handler(model);
            var response = await this.labOrderService.UpdateAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Given  Lab Order name has already been exists with us.");
                case 0:
                    return this.ServerError();
            }
            var auditLogModel = new AuditLogModel
            {
                AccountId = model.ModifiedBy,
                LogTypeId = (int)LogTypes.LabOrders,
                LogFrom = (int)AccountType.Administrator,
                LogDate = DateTime.UtcNow,
                LogDescription = $"{model.Name} Lab Order has been updated successfully.",
                LocationId= Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);
            return this.Success(" Lab Order has been updated successfully.");
        }

        /// <summary>
        /// The delete generalAdvice.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - generalAdvice deleted successfully.
        /// - 409 - generalAdvice can not be deleted.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("delete")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> DeleteAsync([FromBody] LabOrderModel model, [FromHeader] LocationHeader header)
        {
            try
            {
                model = (LabOrderModel)EmptyFilter.Handler(model);
                var response = await this.labOrderService.DeleteAsync(model.LabOrderId);
                if (response == 0)
                {
                    return this.ServerError();
                }
                var auditLogModel = new AuditLogModel
                {
                    AccountId = model.ModifiedBy,
                    LogTypeId = (int)LogTypes.LabOrders,
                    LogFrom = (int)AccountType.Administrator,
                    LogDate = DateTime.UtcNow,
                    LogDescription = $"{model.Name} Lab Order has been deleted successfully.",
                    LocationId= Convert.ToInt32(header.LocationId)
                };
                await this.auditLogServices.LogAsync(auditLogModel);
                return this.Success(" Lab Order has been deleted successfully.");
            }
            catch (NpgsqlException exception)
            {
                return this.ServerError(exception.Message);
            }
        }
    }
}